import os
from io import BytesIO
from django.core.files.base import ContentFile
from PIL import Image
from django.db import models
from django.utils import timezone
from datetime import datetime
from PIL import Image
from io import BytesIO

# =====================================================
# TARAZ OPERATION MODEL
# =====================================================

# 1. Users Table
class TarazUser(models.Model):
    ROLES = [
        ("user", "User"),
        ("admin", "Admin"),
    ]

    id = models.BigAutoField(primary_key=True)
    name = models.CharField(max_length=100)
    email = models.EmailField(max_length=150, unique=True)
    password_hash = models.CharField(max_length=255)

    phone = models.CharField(max_length=20, blank=True, null=True, default="")
    address = models.TextField(blank=True, null=True, default="")

    image = models.ImageField(upload_to="user/image/", blank=True, null=True, default="")

    role = models.CharField(max_length=50, choices=ROLES, default="user")
    is_active = models.BooleanField(default=True)

    created_at = models.DateTimeField(default=datetime.now)
    updated_at = models.DateTimeField(auto_now=True)

    def save(self, *args, **kwargs):
        if self.image and hasattr(self.image, 'file'):
            try:
                img = Image.open(self.image)
                if img.mode in ("RGBA", "P"):
                    img = img.convert("RGB")
                img_io = BytesIO()
                img.save(img_io, format='WEBP', quality=80)
                new_filename = os.path.splitext(self.image.name)[0] + ".webp"
                self.image.save(new_filename, ContentFile(img_io.getvalue()), save=False)
            except Exception as e:
                print("Image conversion failed:", e)
        super().save(*args, **kwargs)

    def __str__(self):
        return self.name


# 2. Product Table
class TarazProduct(models.Model):
    CATEGORY_CHOICES = [
        ("taraaz_fusion_casual", "Taraaz Fusion - Casual wear"),
        ("taraaz_fusion_party", "Taraaz Fusion - Party wear"),
        ("taraaz_fusion_shirt", "Taraaz Fusion - Shirt"),
        ("designers_choice", "Designer's Choice"),
        ("pakistani_wear", "Pakistani Wear"),
        ("indian_wear", "Indian Wear"),
        ("saree", "Saree"),
    ]

    id = models.BigAutoField(primary_key=True)
    name = models.CharField(max_length=255)
    slug = models.SlugField(max_length=255, unique=True, blank=True, null=True)

    # Pricing
    discount_price = models.DecimalField(max_digits=10, decimal_places=2, blank=True, null=True)
    original_price = models.DecimalField(max_digits=10, decimal_places=2, blank=True, null=True)

    # Images (up to 5)
    image1 = models.ImageField(upload_to="clothing/image/", blank=True, null=True, default="")
    image2 = models.ImageField(upload_to="clothing/image/", blank=True, null=True, default="")
    image3 = models.ImageField(upload_to="clothing/image/", blank=True, null=True, default="")
    image4 = models.ImageField(upload_to="clothing/image/", blank=True, null=True, default="")
    image5 = models.ImageField(upload_to="clothing/image/", blank=True, null=True, default="")

    # Product Info
    category = models.CharField(max_length=100, choices=CATEGORY_CHOICES, default="casual_wear", null=True)
    material = models.CharField(max_length=255, blank=True, null=True, default="")
    size = models.JSONField(blank=True, default=list, null=True)  # e.g., ["S", "M", "L", "XL"]
    color = models.JSONField(blank=True, default=list, null=True) # e.g., ["Red", "Blue"]
    fit = models.CharField(max_length=100, blank=True, null=True, default="")
    product_details = models.TextField(blank=True, null=True, default="")
    description = models.TextField(blank=True, null=True, default="")

    # Reviews & FAQs
    reviews = models.JSONField(blank=True, default=list, null=True)
    faqs = models.JSONField(blank=True, default=list, null=True)

    # System timestamps
    created_at = models.DateTimeField(default=datetime.now)
    updated_at = models.DateTimeField(auto_now=True)

    # Business / SEO Info
    b_name = models.CharField(max_length=255, blank=True, default="")
    b_description = models.TextField(blank=True, default="")
    meta_description = models.TextField(blank=True, default="")
    b_meta_description = models.TextField(blank=True, default="")
    tags = models.JSONField(blank=True, default=list, null=True)
    categories = models.JSONField(blank=True, default=list, null=True)
    views = models.IntegerField(default=0)
    stocks = models.BigIntegerField(default=0)
    pin = models.BooleanField(default=False)
    barcode = models.BigIntegerField(unique=True, null=True, blank=True, default=None)
    isPublish = models.BooleanField(default=True)
    isSizeable = models.BooleanField(default=False)
    stocks_size = models.JSONField(blank=True, default=list, null=True)  
    def save(self, *args, **kwargs):
        image_fields = [self.image1, self.image2, self.image3, self.image4, self.image5]
        for field in image_fields:
            if field and hasattr(field, 'file'):
                try:
                    img = Image.open(field)
                    if img.mode in ("RGBA", "P"):
                        img = img.convert("RGB")
                    img_io = BytesIO()
                    img.save(img_io, format='WEBP', quality=80)
                    new_filename = os.path.splitext(field.name)[0] + ".webp"
                    field.save(new_filename, ContentFile(img_io.getvalue()), save=False)
                except Exception as e:
                    print(f"Image conversion failed for {field.name}: {e}")
        super().save(*args, **kwargs)

    def register(self):
        self.save()

    def __str__(self):
        return self.name


# 3. Videos Table
class TarazVideo(models.Model):
    id = models.BigAutoField(primary_key=True)
    title = models.CharField(max_length=255, null=False, blank=False)
    slug = models.SlugField(max_length=255, unique=True, null=True, blank=True)
    thumbnail = models.URLField(max_length=500, blank=True, null=True, default="")
    video_id = models.CharField(max_length=500, blank=True, null=True, default="")
    pin = models.BooleanField(default=False)
    system_date = models.DateTimeField(default=datetime.now, blank=True, null=True)

    def register(self):
        self.save()


# 4. Order Table
class TarazOrder(models.Model):
    ORDER_STATUS_CHOICES = [
        ("pending", "Pending"),
        ("confirmed", "Confirmed"),
        ("shipped", "Shipped"),
        ("delivered", "Delivered"),
        ("cancelled", "Cancelled"),
    ]

    PAYMENT_STATUS_CHOICES = [
        ("pending", "Pending"),
        ("paid", "Paid"),
        ("failed", "Failed"),
    ]

    id = models.BigAutoField(primary_key=True)
    user_id = models.CharField(max_length=20, null=False, blank=False)
    customer_name = models.CharField(max_length=100, null=False, blank=False)
    customer_email = models.EmailField(max_length=150, blank=True, null=True, default="")
    customer_phone = models.CharField(max_length=20, blank=True, null=True, default="")
    delivery_address = models.TextField(blank=True, null=True, default="")

    items_data = models.JSONField(blank=True, default=list)
    total_amount = models.DecimalField(max_digits=10, decimal_places=2, null=False)
    order_status = models.CharField(max_length=50, choices=ORDER_STATUS_CHOICES, default="pending")

    payment_amount = models.DecimalField(max_digits=10, decimal_places=2, null=True, blank=True)
    payment_currency = models.CharField(max_length=10, default="BDT")
    payment_method = models.CharField(max_length=50, null=True, blank=True)
    payment_status = models.CharField(max_length=50, choices=PAYMENT_STATUS_CHOICES, default="pending")
    transaction_id = models.CharField(max_length=100, blank=True, null=True, default="")
    provider = models.CharField(max_length=100, blank=True, null=True, default="")
    paid_at = models.DateTimeField(blank=True, null=True)

    created_at = models.DateTimeField(default=datetime.now, blank=True, null=True)
    updated_at = models.DateTimeField(auto_now=True)
    system_date = models.DateTimeField(default=datetime.now, blank=True, null=True)

    def register(self):
        self.save()


# Blog Table
class TarazBlog(models.Model):
    STATUS_CHOICES = [
        ("draft", "Draft"),
        ("published", "Published"),
    ]

    id = models.BigAutoField(primary_key=True)
    title = models.CharField(max_length=255)
    slug = models.SlugField(max_length=255, unique=True, blank=True, null=True)
    content = models.TextField()
    excerpt = models.TextField(blank=True, null=True, default="")

    # Images
    cover_image = models.ImageField(upload_to="blog/images/", blank=True, null=True, default="")
    author_image = models.ImageField(upload_to="blog/authors/", blank=True, null=True, default="")

    author_name = models.CharField(max_length=100, blank=True, null=True, default="Admin")
    tags = models.JSONField(blank=True, default=list)

    meta_title = models.CharField(max_length=255, blank=True, null=True, default="")
    meta_description = models.CharField(max_length=1500, blank=True, null=True, default="")
    b_meta_description = models.TextField(blank=True, null=True, default="")

    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default="draft")
    published_at = models.DateTimeField(blank=True, null=True)

    created_at = models.DateTimeField(default=datetime.now, blank=True, null=True)
    updated_at = models.DateTimeField(auto_now=True)
    system_date = models.DateTimeField(default=datetime.now, blank=True, null=True)

    b_title = models.CharField(max_length=255, blank=True, default="")
    b_content = models.TextField(blank=True, default="")

    def save(self, *args, **kwargs):
        image_fields = [self.cover_image, self.author_image]

        for field in image_fields:
            if field and hasattr(field, 'file'):
                try:
                    img = Image.open(field)
                    if img.mode in ("RGBA", "P"):
                        img = img.convert("RGB")
                    img_io = BytesIO()
                    img.save(img_io, format='WEBP', quality=80)
                    new_filename = os.path.splitext(field.name)[0] + ".webp"
                    field.save(new_filename, ContentFile(img_io.getvalue()), save=False)
                except Exception as e:
                    print(f"Image conversion failed for {field.name}: {e}")

        super().save(*args, **kwargs)

    def register(self):
        self.save()

    def __str__(self):
        return self.title


# Contact Form
class TarazContact(models.Model):
    id = models.BigAutoField(primary_key=True)
    name = models.CharField(max_length=100, null=False, blank=False)
    email = models.EmailField(max_length=150, null=False, blank=False)
    phone = models.CharField(max_length=20, blank=True, null=True, default="")
    subject = models.CharField(max_length=200, blank=True, null=True, default="")
    message = models.TextField(null=False, blank=False)
    created_at = models.DateTimeField(default=datetime.now, blank=True, null=True)
    system_date = models.DateTimeField(default=datetime.now, blank=True, null=True)

    def register(self):
        self.save()


# Company Info
class TarazCompanyContact(models.Model):
    id = models.BigAutoField(primary_key=True)
    company_name = models.CharField(max_length=255, blank=True, null=True, default="")
    phone = models.CharField(max_length=50, blank=True, null=True, default="")
    email = models.EmailField(max_length=150, blank=True, null=True, default="")
    website = models.URLField(max_length=255, blank=True, null=True, default="")
    address = models.TextField(blank=True, null=True, default="")
    city = models.CharField(max_length=100, blank=True, null=True, default="")
    country = models.CharField(max_length=100, blank=True, null=True, default="")
    facebook = models.URLField(max_length=255, blank=True, null=True, default="")
    instagram = models.URLField(max_length=255, blank=True, null=True, default="")
    youtube = models.URLField(max_length=255, blank=True, null=True, default="")
    linkedin = models.URLField(max_length=255, blank=True, null=True, default="")
    whatsapp = models.URLField(max_length=255, blank=True, null=True, default="")
    tiktok = models.URLField(max_length=255, blank=True, null=True, default="")
    imo = models.URLField(max_length=255, blank=True, null=True, default="")
    twitter = models.URLField(max_length=255, blank=True, null=True, default="")
    map = models.TextField(blank=True, null=True, default="")
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    system_date = models.DateTimeField(default=datetime.now, blank=True, null=True)
    b_company_name = models.CharField(max_length=255, blank=True, null=True, default="")
    b_phone = models.CharField(max_length=50, blank=True, null=True, default="")
    b_address = models.TextField(blank=True, null=True, default="")

    def register(self):
        self.save()


# Banners
class TarazBanner(models.Model):
    id = models.BigAutoField(primary_key=True)
    name = models.CharField(max_length=255)
    slug = models.SlugField(max_length=255, unique=True, blank=True)
    image = models.ImageField(upload_to="banner/image/", blank=True, null=True, default="")
    side_image = models.ImageField(upload_to="banner/image/", blank=True, null=True, default="")
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def save(self, *args, **kwargs):
        image_fields = [self.image, self.side_image]

        for field in image_fields:
            if field and hasattr(field, 'file'):
                try:
                    img = Image.open(field)
                    if img.mode in ("RGBA", "P"):
                        img = img.convert("RGB")
                    img_io = BytesIO()
                    img.save(img_io, format='WEBP', quality=80)
                    new_filename = os.path.splitext(field.name)[0] + ".webp"
                    field.save(new_filename, ContentFile(img_io.getvalue()), save=False)
                except Exception as e:
                    print(f"Image conversion failed for {field.name}: {e}")

        super().save(*args, **kwargs)

    def register(self):
        self.save()

    def __str__(self):
        return self.name


class TarazBottomBanner(models.Model):
    id = models.BigAutoField(primary_key=True)
    name = models.CharField(max_length=255, blank=False, null=False)
    slug = models.SlugField(max_length=255, unique=True, blank=True)
    image = models.ImageField(upload_to="banner/image/", blank=True, null=True, default="")
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def save(self, *args, **kwargs):
        if self.image and hasattr(self.image, 'file'):
            try:
                img = Image.open(self.image)
                if img.mode in ("RGBA", "P"):
                    img = img.convert("RGB")
                img_io = BytesIO()
                img.save(img_io, format='WEBP', quality=80)
                new_filename = os.path.splitext(self.image.name)[0] + ".webp"
                self.image.save(new_filename, ContentFile(img_io.getvalue()), save=False)
            except Exception as e:
                print(f"Image conversion failed for {self.image.name}: {e}")

        super().save(*args, **kwargs)

    def register(self):
        self.save()

    def __str__(self):
        return self.name


# Website Titles
class TarazWebsiteTitle(models.Model):
    id = models.BigAutoField(primary_key=True)
    page_name = models.CharField(max_length=255, blank=False, null=False)
    title = models.CharField(max_length=255, blank=False, null=False)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    b_title = models.CharField(max_length=255, blank=True, null=False, default="")

    def register(self):
        self.save()


# FAQs
class TarazFAQ(models.Model):
    id = models.BigAutoField(primary_key=True)
    title = models.CharField(max_length=255, blank=False, null=False)
    b_title = models.CharField(max_length=255, blank=False, null=False)
    items = models.JSONField(default=list, blank=False, null=False)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def register(self):
        self.save()


# Hit Counter
class TarazHitcounter(models.Model):
    id = models.BigAutoField(primary_key=True)
    total_visitor = models.IntegerField(blank=True, null=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def register(self):
        self.save()


# Category
class TarazCategory(models.Model):
    id = models.BigAutoField(primary_key=True)
    name = models.CharField(max_length=255, blank=False, null=False)
    value = models.CharField(max_length=255, blank=False, null=False)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def register(self):
        self.save()
